//
//  RevenueFlo.swift
//  MicroQRSampleApp
//
//  Created by Velu on 09/12/24.
//

import Foundation
import UIKit
import DeviceCheck
import CryptoKit
import Security
import React
import WebKit

// MARK: - Constants
private struct RFConstants {
    static let contentType = "application/json"
    static let fetchOfferCampaigns = "inapp-offer-campaigns"
    static let trackOfferConversion = "inapp-offer-conversions"
    #if DEBUG
    static let attestKeyIdKey = "AttestKeyIdDebug"
    #else
    static let attestKeyIdKey = "AttestKeyId"
    #endif
}

// MARK: - Errors
enum RFError: Error {
    case attestVerificationFailed
    case attestNotSupported
    case assertionFailed
    case invalidTeamId
    case invalidResponseFormat
    case noActiveCampaignsFound
    
    var localizedDescription: String {
        switch self {
        case .attestVerificationFailed:
            return "Attestation verification failed."
        case .attestNotSupported:
            return "App Attest is not supported on this device."
        case .assertionFailed:
            return "Assertion generation failed."
        case .invalidTeamId:
            return "The provided team ID is invalid."
        case .invalidResponseFormat:
            return "The response format from the server is invalid."
        case .noActiveCampaignsFound:
            return "No active offer campaigns found for the provided appId."
        }
    }
}

// MARK: - Logger Utility
final class RFLogger {
    
    // MARK: - Logging Levels
    enum LogLevel: String {
        case debug = "DEBUG"
        case info = "INFO"
        case warning = "WARNING"
        case error = "ERROR"
    }
    
    // MARK: - Log Method
    static func log(_ message: String, level: LogLevel = .info, function: String = #function, file: String = #file, line: Int = #line) {
        guard RevenueFlo.shared.logsEnabled else { return }
        print("[RevenueFlo] [\(level.rawValue)] \(message)")
    }
}


// MARK: - RFOfferCampaign Model
public struct RFOfferCampaign: Decodable {
    let id: String
    let name: String
    let popupTitle: String?
    let popupDescription: String?
    let popupCTA: String?
    let popupColorCode: String?
    let popupImage: String?
    let showPopupCloseIcon: Bool
    let status: String
    let customCode: String?
    let customerEligibilities: [String]?
    let offer: String?
    let offerId: String?
    let shortLink: String?
    let template: String?
    let theme: String?
}

// MARK: - RFNetworkManager
class RFNetworkManager {
    private let baseURL: String
    private let bundleID: String
    private let teamID: String
    
    init(baseURL: String, bundleID: String, teamID: String) {
        self.baseURL = baseURL
        self.bundleID = bundleID
        self.teamID = teamID
    }
    
    func fetchOfferCampaigns() async throws -> RFOfferCampaign {
        let challenge = try await retrieveChallenge()
        let payload = try JSONEncoder().encode([
            "bundleIdentifier": bundleID,
            "challenge": challenge,
        ])
        
        let assertion = try await createAssertion(payload)
        
        var request = URLRequest(url: url(RFConstants.fetchOfferCampaigns))
        request.httpMethod = "POST"
        request.httpBody = payload
        request.setValue(RFConstants.contentType, forHTTPHeaderField: "Content-Type")
        request.setValue(assertion, forHTTPHeaderField: "authentication")
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse else {
            throw RFError.invalidResponseFormat
        }
        
        if httpResponse.statusCode == 401 {
            //            UserDefaults.standard.removeObject(forKey: RFConstants.attestKeyIdKey)
            RevenueFlo.shared.storageManager.delete(key: RFConstants.attestKeyIdKey)
            throw RFError.assertionFailed
        } else if httpResponse.statusCode == 404 {
            RFLogger.log("No active offer campaigns found. Please check your project's RevenueFlo dashboard.", level: .error)
            throw RFError.noActiveCampaignsFound
        } else {
            let jsonObject = try JSONSerialization.jsonObject(with: data)
            guard let json = jsonObject as? [String: Any] else {
                RFLogger.log("The received data is not in the expected format.", level: .error)
                throw RFError.invalidResponseFormat
            }
            
            let campaignData = json
            return try JSONDecoder().decode(RFOfferCampaign.self, from: try JSONSerialization.data(withJSONObject: campaignData))
        }
    }
    
    func trackOfferConversion(params: [String: Any?]) async throws -> Bool {
        let challenge = try await retrieveChallenge()
        var payloadDict: [String: Any?] = [
            "challenge": challenge,
            "offer": params["offer"] ?? nil,
            "transaction_id": params["transaction_id"] ?? nil,
            "original_transaction_id": params["original_transaction_id"] ?? nil,
            "product_id": params["product_id"] ?? nil,
            "bundleId": params["bundleId"] ?? nil,
            "price": params["price"] ?? nil,
            "currencyCode": params["currencyCode"] ?? nil,
            "purchasedAt": params["purchasedAt"] ?? nil,
            "expirationAt": params["expirationAt"] ?? nil,
        ]
        
        if let offerCode = params["offerCode"] {
            payloadDict["offerCode"] = offerCode
        }
        
        let payload = try JSONSerialization.data(withJSONObject: payloadDict, options: [])
        let assertion = try await createAssertion(payload)
        
        var request = URLRequest(url: url(RFConstants.trackOfferConversion))
        request.httpMethod = "POST"
        request.httpBody = payload
        request.setValue(RFConstants.contentType, forHTTPHeaderField: "Content-Type")
        request.setValue(assertion, forHTTPHeaderField: "authentication")
        
        let (data, response) = try await URLSession.shared.data(for: request)
        
        guard let httpResponse = response as? HTTPURLResponse else {
            throw RFError.invalidResponseFormat
        }
        
        if httpResponse.statusCode == 401 {
            RevenueFlo.shared.storageManager.delete(key: RFConstants.attestKeyIdKey)
            throw RFError.assertionFailed
        } else {
            let jsonObject = try JSONSerialization.jsonObject(with: data)
            guard let _ = jsonObject as? [String: Any] else {
                RFLogger.log("The received data is not in the expected format.", level: .error)
                throw RFError.invalidResponseFormat
            }
            
            return true
        }
    }
    
    private func url(_ target: String) -> URL {
        return URL(string: "\(baseURL)/\(target)")!
    }
    
    private func retrieveChallenge() async throws -> String {
        let (data, _) = try await URLSession.shared.data(from: url("apple-app-attest/challenge"))
        let json = try JSONDecoder().decode([String: String].self, from: data)
        guard let challenge = json["challenge"] else {
            throw RFError.invalidResponseFormat
        }
        return challenge
    }
    
    private func createAssertion(_ payload: Data) async throws -> String {
        do {
            var keyId = RevenueFlo.shared.storageManager.retrieve(for: RFConstants.attestKeyIdKey)
            if keyId == nil {
                keyId = try await attestKey()
            }
            let hash = Data(SHA256.hash(data: payload))
            let service = DCAppAttestService.shared
            let assertion = try await service.generateAssertion(keyId!, clientDataHash: hash)
            return try JSONEncoder().encode([
                "keyId": keyId,
                "assertion": assertion.base64EncodedString(),
            ]).base64EncodedString()
        } catch {
            RevenueFlo.shared.storageManager.delete(key: RFConstants.attestKeyIdKey)
            throw error
        }
    }
    
    private func attestKey() async throws -> String {
        let service = DCAppAttestService.shared
        if service.isSupported {
            let challenge = try await retrieveChallenge()
            let keyId = try await service.generateKey()
            let clientDataHash = Data(SHA256.hash(data: challenge.data(using: .utf8)!))
            let attestation = try await service.attestKey(keyId, clientDataHash: clientDataHash)
            
            var request = URLRequest(url: url("apple-app-attest/verify"))
            request.httpMethod = "POST"
            request.httpBody = try JSONEncoder().encode(
                [
                    "bundleIdentifier": bundleID,
                    "teamIdentifier": teamID,
                    "keyId": keyId,
                    "challenge": challenge,
                    "attestation": attestation.base64EncodedString(),
                ]
            )
            request.setValue(RFConstants.contentType, forHTTPHeaderField: "Content-Type")
            
            let (_, response) = try await URLSession.shared.data(for: request)
            
            if let httpResponse = response as? HTTPURLResponse, httpResponse.statusCode == 204 {
//                UserDefaults.standard.set(keyId, forKey: RFConstants.attestKeyIdKey)
                RevenueFlo.shared.storageManager.save(value: keyId, for: RFConstants.attestKeyIdKey)
                return keyId
            }
            
            throw RFError.attestVerificationFailed
        }
        throw RFError.attestNotSupported
    }
}

protocol RevenueFloActionDelegate: AnyObject {
    func offerDidPresent(_ offer: String)
    func offerDidClose(_ offer: String)
    func offerPrimaryButtonDidClick(_ offer: String)
}

// MARK: - RevenueFlo
@objc(RevenueFlo) // For React Native
@available(iOS 15.0, *)
class RevenueFlo: RCTEventEmitter { // For React Native
    public static let shared = RevenueFlo()
    private var offerCampaigns: [RFOfferCampaign] = []
    private var currentOfferCampaign: RFOfferCampaign?
    private var networkManager: RFNetworkManager!
    public var storageManager: RFStorageManager!
    
    public static let plistReader = RFPlistReader(plistName: "RevenueFlo-Info")
    private var isConfigured = false
    public var appstoreID: String?
    public var logsEnabled: Bool = true
    private var customerType: CustomerType?
    var customerCountryCode = "USA"
    var customerCurrency = "$"
    
//    private init() {}
    
    // React Native related start
    override func supportedEvents() -> [String]! {
      return ["offerDidPresent", "offerDidClose", "offerPrimaryButtonDidClick"]
    }
    
    @objc override static func requiresMainQueueSetup() -> Bool {
      return true
    }

    @objc func configureForRN() {
      RevenueFlo.configure()
      RevenueFlo.shared.fetchOfferCampaigns()
    }

    @objc func presentOfferForRN(_ delay: Double = 0) {
      DispatchQueue.main.async {
              // Fetch the root view controller
              guard let scene = UIApplication.shared.connectedScenes.first(where: { $0.activationState == .foregroundActive }),
                    let windowScene = scene as? UIWindowScene,
                    let rootVC = windowScene.windows.first(where: { $0.isKeyWindow })?.rootViewController else {
                  print("Error: Root view controller not found.")
                  return
              }
          
              RevenueFlo.shared.presentOffer(from: rootVC, delay: delay, delegate: self)
          }
    }
    // React Native related end
    // Configure the SDK
    public static func configure() {
        guard !shared.isConfigured else { return }
        
        // Read configuration values from the plist
        guard let teamId = plistReader.string(forKey: "TEAM_ID") else {
            fatalError("TEAM_ID not found in RevenueFlo-Info.plist")
        }
        guard let serverURL = plistReader.string(forKey: "SERVER_URL") else {
            fatalError("SERVER_URL not found in RevenueFlo-Info.plist")
        }
        
        guard let appstoreID = plistReader.string(forKey: "APPSTORE_ID") else {
            fatalError("APPSTORE_ID not found in RevenueFlo-Info.plist")
        }
        
        // Set up network manager
        shared.networkManager = RFNetworkManager(baseURL: serverURL, bundleID: Bundle.main.bundleIdentifier!, teamID: teamId)
        
        shared.storageManager = RFStorageManager.shared
        
        // Mark as configured
        shared.isConfigured = true
        shared.appstoreID = appstoreID
        
        // Set up app state listener to fetch offer campaigns when the app enters the foreground
        shared.configureAppStateListener()
        
    }
    
    // Listen for app state changes (when app comes to foreground)
    func configureAppStateListener() {
        NotificationCenter.default.addObserver(self, selector: #selector(fetchOfferCampaigns), name: UIApplication.willEnterForegroundNotification, object: nil)
    }
    
    public func getOfferCampaigns() -> [RFOfferCampaign] {
        return offerCampaigns
    }
    
    public func getCurrentOfferCampaign() -> RFOfferCampaign? {
        return currentOfferCampaign
    }
    
    @objc private func fetchOfferCampaigns() {
        
        Task {
            let maxRetries = 2

            for attempt in 1...maxRetries {
                do {
                    let campaign = try await networkManager.fetchOfferCampaigns()
                    currentOfferCampaign = campaign
                    RFLogger.log("currentOfferCampaign: \(currentOfferCampaign!.name)")
                
                    let (countryCode, currencySymbol) = await getStorefrontInfo()
                    RFLogger.log("App Store Country: \(countryCode ?? "Unknown")")
                    RFLogger.log("Currency symbol: \(currencySymbol ?? "Unknown")")
                    customerCountryCode = countryCode ?? "USA"
                    customerCurrency = currencySymbol ?? "$"
                
                    let subscriptionStatus = await RFSubscriptionManager.shared.checkSubscriptionStatus()
                    RFLogger.log("Subscription Status: \(subscriptionStatus)")
                    customerType = subscriptionStatus
                
                    // Success - break out of retry loop
                    break

                } catch {
                    RFLogger.log("Error fetching offer campaigns - Attempt \(attempt)/\(maxRetries): \(error)", level: .error)
                    // If this is not the last attempt, continue to retry
                    if attempt < maxRetries {
                        continue
                    } else {
                        // Last attempt failed, log final error
                        RFLogger.log("Failed to fetch offer campaigns after \(maxRetries) attempts: \(error)", level: .error)
                    }
                }
            }
        }
        
        
    }
    
    private func getStorefrontInfo() async -> (countryCode: String?, currencySymbol: String?) {
        if let storefront = await StoreKit.Storefront.current {
            let countryCode = storefront.countryCode
            // Create a locale identifier based on the country code
            let localeIdentifier = Locale.identifier(fromComponents: [NSLocale.Key.countryCode.rawValue: countryCode])
            let locale = Locale(identifier: localeIdentifier)
            
            // Get the currency symbol for that locale
            let currencySymbol = locale.currencySymbol ?? "Unknown Symbol"
            return (countryCode, currencySymbol)
        }
        return (nil, nil)
    }
    
    func trackOfferConversion(transaction: Transaction) {
        Task {
            
            var offer: String?
            if #available(iOS 17.2, *) {
                offer = transaction.offer?.id
            } else {
                // Fallback on earlier versions
                offer = transaction.offerID
            }
            
            guard let claimedOffer = offer else {
                return
            }
            
            if (checkOfferCreatedViaRevenueFlo(offer: claimedOffer)) {
    
                updateCustomerType(offer: offer!)
                let transactionId = "\(transaction.id)"
                
                if let _ = RevenueFlo.shared.storageManager.retrieve(for: transactionId) {
                    RFLogger.log("New in-app offer conversion has been already tracked!")
                    return
                }
                
                var params = [
                    "offer": offer as Any,
                    "transaction_id": transactionId,
                    "original_transaction_id": "\(transaction.originalID)",
                    "product_id": transaction.productID,
                    "bundleId":  transaction.appBundleID,
                    "price": transaction.price,
                    "purchasedAt": RevenueFlo.shared.stringFromDate(date: transaction.purchaseDate),
                    "expirationAt": transaction.expirationDate != nil ? RevenueFlo.shared.stringFromDate(date: transaction.expirationDate!) : nil
                ]
                
                if #available(iOS 16.0, *) {
                    params["currencyCode"] = transaction.currency?.identifier
                } else {
                    // Fallback on earlier versions
                    params["currencyCode"] = transaction.currencyCode
                }
                
                do {
                    let success = try await networkManager.trackOfferConversion(params: params)
                    if (success) {
                        RFLogger.log("New in-app offer conversion has been tracked!")
                        RevenueFlo.shared.storageManager.save(value: transaction.productID, for: transactionId)
                    }
                } catch {
                    RFLogger.log("Error tracking offer conversion: \(error)", level: .error)
                }
                
                
                
            } else {
                RFLogger.log("Active subscription is not associated with RevenueFlo.com campaigns.")
            }
            
            
        }
    }
    
    func updateCustomerType(offer: String) {
        if let offerCampaign = currentOfferCampaign {
            let encodedOfferId = base62Encode(offerCampaign.offerId!)
            if (offer.contains(encodedOfferId)){
                customerType = .claimed
            }
        }
    }
    
    func checkOfferCreatedViaRevenueFlo(offer: String) -> Bool {
        if (offer.contains("@RevenueFlo")) {
            return true
        }
        return false
    }
    
    func stringFromDate(date: Date) -> String {
        let dateFormatter = ISO8601DateFormatter()
            dateFormatter.formatOptions = [.withInternetDateTime, .withFractionalSeconds]
        return dateFormatter.string(from: date)
    }
    
    func presentOffer(from viewController: UIViewController, delay: Double = 0, delegate: RevenueFloActionDelegate? = nil) {
        DispatchQueue.main.asyncAfter(deadline: .now() + delay) {
            if let offer = RevenueFlo.shared.currentOfferCampaign, let customerType =  RevenueFlo.shared.customerType {
                if let eligibilities = offer.customerEligibilities {
                    if(eligibilities.contains(customerType.rawValue)) {
                        RFLogger.log("Customer Type: \(customerType)")
                        let offerPopupVC = RevenueFloOfferVC(
                            currentOffer: offer,
                            delegate: delegate,
                            customerCountryCode: self.customerCountryCode,
                            customerCurrency: self.customerCurrency
                        )
                        offerPopupVC.modalPresentationStyle = .overFullScreen
                        offerPopupVC.modalTransitionStyle = .crossDissolve
                        viewController.present(offerPopupVC, animated: true, completion: nil)
                    } else if customerType == .claimed {
                        RFLogger.log("Cutomer has claimed this offer already.")
                    } else {
                        RFLogger.log("Cutomer is not eligible for this offer campaign.")
                    }
                }
            } else {
                RFLogger.log("No active offer campaigns found!")
            }
        }
    }
    
    // For SwiftUI Apps
    func topViewController() -> UIViewController? {
        guard let windowScene = UIApplication.shared.connectedScenes
                .first(where: { $0.activationState == .foregroundActive }) as? UIWindowScene,
              let window = windowScene.windows.first(where: { $0.isKeyWindow }),
              var topController = window.rootViewController else {
            return nil
        }
        while let presentedViewController = topController.presentedViewController {
            topController = presentedViewController
        }
        return topController
    }

    func showOffer(delay: Double = 0, delegate: RevenueFloActionDelegate? = nil) {
        DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) {
            guard let topVC = self.topViewController() else {
                print("No top view controller found.")
                return
            }
            RevenueFlo.shared.presentOffer(from: topVC, delay: delay, delegate: delegate)
        }
    }
}

// React Native related start
extension RevenueFlo: RevenueFloActionDelegate {
  
    func offerDidPresent(_ offer: String) {
      sendEvent(withName: "offerDidPresent", body: offer)
    }
    
    func offerDidClose(_ offer: String) {
      sendEvent(withName: "offerDidClose", body: offer)
    }
    
    func offerPrimaryButtonDidClick(_ offer: String) {
      sendEvent(withName: "offerPrimaryButtonDidClick", body: offer)
    }
}
// React Native related end

class RevenueFloOfferVC: UIViewController, WKNavigationDelegate {
    
    weak var delegate: RevenueFloActionDelegate?
    private var currentOffer: RFOfferCampaign!
    private var webView: WKWebView?
    private var customerCountryCode: String
    private var customerCurrency: String
    
    // Initialize with dynamic title, description, and an optional image
    init(currentOffer: RFOfferCampaign, delegate: RevenueFloActionDelegate?, customerCountryCode: String, customerCurrency: String) {
        self.currentOffer = currentOffer
        self.delegate = delegate
        self.customerCountryCode = customerCountryCode
        self.customerCurrency = customerCurrency
        super.init(nibName: nil, bundle: nil)
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        if let template = currentOffer.template, !template.isEmpty, template != "template1" {
            setupOfferWebview()
        } else {
            setupOfferPopupUI()
        }
    }
    
    override func viewDidLayoutSubviews() {
        super.viewDidLayoutSubviews()
    }
    
    private func updateWebviewFrame() {
        guard let webView = webView else { return }
        var frame = view.bounds
        // Extend upward to cover status bar area
        frame.origin.y = -view.safeAreaInsets.top
        frame.size.height += view.safeAreaInsets.top
        webView.frame = frame
    }
    
    private func setupOfferPopupUI() {
        view.backgroundColor = UIColor.black.withAlphaComponent(0.5)
        
        // Popup container view
        let popupView = UIView()
        popupView.backgroundColor = .white
        popupView.layer.cornerRadius = 10
        popupView.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(popupView)
        
        // Close button (X)
        let closeButton = UIButton(type: .system)
        closeButton.setTitle("✕", for: .normal)
        closeButton.titleLabel?.font = UIFont.systemFont(ofSize: 20)
        closeButton.setTitleColor(.darkGray, for: .normal)
        closeButton.addTarget(self, action: #selector(closePopup), for: .touchUpInside)
        closeButton.translatesAutoresizingMaskIntoConstraints = false
        popupView.addSubview(closeButton)
        
        // ImageView for icon (optional)
        let iconImageView = UIImageView()
        iconImageView.image = UIImage(systemName: "gift.fill")
        if let imageName = currentOffer.popupImage {
            if let image = UIImage(systemName: imageName.replacingOccurrences(of: "_", with: ".")) {
                iconImageView.image = image
            }
            iconImageView.tintColor = UIColor(hex: currentOffer.popupColorCode ?? "#0272D4")
        } else {
            iconImageView.tintColor = UIColor(hex: currentOffer.popupColorCode ?? "#0272D4")
        }
        iconImageView.contentMode = .scaleAspectFit
        iconImageView.translatesAutoresizingMaskIntoConstraints = false
        popupView.addSubview(iconImageView)
        
        // Title Label
        let titleLabel = UILabel()
        titleLabel.numberOfLines = 0
        titleLabel.lineBreakMode = .byWordWrapping
        titleLabel.text = currentOffer.popupTitle // Dynamic title
        titleLabel.textColor = UIColor.darkGray
        titleLabel.font = UIFont.boldSystemFont(ofSize: 24)
        titleLabel.textAlignment = .center
        titleLabel.translatesAutoresizingMaskIntoConstraints = false
        popupView.addSubview(titleLabel)
        
        // Description Label
        let descriptionLabel = UILabel()
        descriptionLabel.textColor = UIColor.darkGray
        descriptionLabel.text = currentOffer.popupDescription // Dynamic description
        descriptionLabel.font = UIFont.systemFont(ofSize: 16)
        descriptionLabel.numberOfLines = 0
        descriptionLabel.lineBreakMode = .byWordWrapping
        descriptionLabel.textAlignment = .center
        descriptionLabel.translatesAutoresizingMaskIntoConstraints = false
        popupView.addSubview(descriptionLabel)
        
        // Claim Button
        let claimButton = UIButton(type: .system)
        claimButton.setTitle(currentOffer.popupCTA ?? "Redeem Offer", for: .normal)
        claimButton.backgroundColor = UIColor(hex: currentOffer.popupColorCode ?? "#0272D4")
        claimButton.setTitleColor(.white, for: .normal)
        claimButton.titleLabel?.font = UIFont.boldSystemFont(ofSize: 18)
        claimButton.layer.cornerRadius = 8
        claimButton.addTarget(self, action: #selector(claimOffer), for: .touchUpInside)
        claimButton.translatesAutoresizingMaskIntoConstraints = false
        popupView.addSubview(claimButton)
        
        // Auto Layout Constraints
        NSLayoutConstraint.activate([
            // Popup View Constraints
            popupView.centerYAnchor.constraint(equalTo: view.centerYAnchor),
            popupView.centerXAnchor.constraint(equalTo: view.centerXAnchor),
            popupView.widthAnchor.constraint(equalTo: view.widthAnchor, multiplier: 0.8),
            popupView.heightAnchor.constraint(lessThanOrEqualTo: view.heightAnchor, constant: -100),
            
            // Close Button Constraints
            closeButton.topAnchor.constraint(equalTo: popupView.topAnchor, constant: 10),
            closeButton.trailingAnchor.constraint(equalTo: popupView.trailingAnchor, constant: -10),
            
            // Icon ImageView Constraints
            iconImageView.centerXAnchor.constraint(equalTo: popupView.centerXAnchor),
            iconImageView.topAnchor.constraint(equalTo: popupView.topAnchor, constant: 40),
            iconImageView.heightAnchor.constraint(equalToConstant: 40),
            iconImageView.widthAnchor.constraint(equalToConstant: 40),
            
            // Title Label Constraints
            titleLabel.leadingAnchor.constraint(equalTo: popupView.leadingAnchor, constant: 20),
            titleLabel.topAnchor.constraint(equalTo: iconImageView.bottomAnchor, constant: 10),
            titleLabel.centerXAnchor.constraint(equalTo: popupView.centerXAnchor),
            
            // Description Label Constraints
            descriptionLabel.topAnchor.constraint(equalTo: titleLabel.bottomAnchor, constant: 10),
            descriptionLabel.leadingAnchor.constraint(equalTo: popupView.leadingAnchor, constant: 20),
            descriptionLabel.trailingAnchor.constraint(equalTo: popupView.trailingAnchor, constant: -20),
            
            // Claim Button Constraints
            claimButton.topAnchor.constraint(equalTo: descriptionLabel.bottomAnchor, constant: 20),
            claimButton.leadingAnchor.constraint(equalTo: popupView.leadingAnchor, constant: 20),
            claimButton.trailingAnchor.constraint(equalTo: popupView.trailingAnchor, constant: -20),
            claimButton.heightAnchor.constraint(equalToConstant: 50),
            claimButton.bottomAnchor.constraint(equalTo: popupView.bottomAnchor, constant: -20)
        ])
        
        delegate?.offerDidPresent(currentOffer.name)
    }
    
    private func setupOfferWebview() {
//        view.backgroundColor = UIColor.white
        
        // Create WKWebView
        let webView = WKWebView()
        self.webView = webView
        // Set navigation delegate to handle link clicks
        webView.navigationDelegate = self
        webView.configuration.userContentController.add(self, name: "openSafari")
        webView.configuration.userContentController.add(self, name: "openPrimaryCTA")
        // Remove content insets to prevent white space
        webView.scrollView.contentInsetAdjustmentBehavior = .never
        webView.scrollView.contentInset = .zero
        webView.scrollView.scrollIndicatorInsets = .zero
        view.addSubview(webView)
        
        // Set webview frame to extend to device edges (including behind status bar)
        updateWebviewFrame()
        
        // Close button (X)
        let closeButton = UIButton(type: .system)
        closeButton.setTitle("✕", for: .normal)
        closeButton.titleLabel?.font = UIFont.systemFont(ofSize: 24, weight: .bold)
        closeButton.setTitleColor(.darkGray, for: .normal)
        closeButton.backgroundColor = UIColor.white.withAlphaComponent(0.9)
        closeButton.layer.cornerRadius = 20
        closeButton.addTarget(self, action: #selector(closePopup), for: .touchUpInside)
        closeButton.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(closeButton)
        
        // Auto Layout Constraints for close button only
        NSLayoutConstraint.activate([
            // Close Button Constraints - positioned in safe area
            closeButton.topAnchor.constraint(equalTo: view.safeAreaLayoutGuide.topAnchor, constant: 10),
            closeButton.trailingAnchor.constraint(equalTo: view.trailingAnchor, constant: -20),
            closeButton.widthAnchor.constraint(equalToConstant: 40),
            closeButton.heightAnchor.constraint(equalToConstant: 40)
        ])
        
        // Build and load URL
        if let url = buildOfferURL() {
            let request = URLRequest(url: url)
            webView.load(request)
        } else {
            RFLogger.log("Failed to build offer URL", level: .warning)
        }
        
        delegate?.offerDidPresent(currentOffer.name)
    }
    
    private func buildOfferURL() -> URL? {
        // Read offersURL from plist
        let plistReader = RevenueFlo.plistReader
        guard let baseURL = plistReader.string(forKey: "OFFERS_URL") else {
            RFLogger.log("OFFERS_URL not found in RevenueFlo-Info.plist", level: .error)
            return nil
        }
        
        // Guard check for shortLink
        guard let shortLink = currentOffer.shortLink, !shortLink.isEmpty else {
            RFLogger.log("shortLink not found in current offer", level: .error)
            return nil
        }
        
        // Build URL components
        var urlString = baseURL
        // Ensure base URL doesn't end with /
        if urlString.hasSuffix("/") {
            urlString.removeLast()
        }
        urlString += "/in-app/\(shortLink)"
        
        // Build query parameters
        var queryItems: [String] = []
        queryItems.append("country=\(customerCountryCode)")
        queryItems.append("currencySymbol=\(customerCurrency)")
        
        if let theme = currentOffer.theme {
            // URL encode the theme
            if let encodedTheme = theme.addingPercentEncoding(withAllowedCharacters: .urlQueryAllowed) {
                queryItems.append("theme=\(encodedTheme)")
            } else {
                queryItems.append("theme=\(theme)")
            }
        }
        
        // Append query string
        if !queryItems.isEmpty {
            urlString += "?" + queryItems.joined(separator: "&")
        }
        
        return URL(string: urlString)
    }
    
    // MARK: - WKNavigationDelegate
    func webView(_ webView: WKWebView, decidePolicyFor navigationAction: WKNavigationAction, decisionHandler: @escaping (WKNavigationActionPolicy) -> Void) {
        // Allow all navigation within the webview
        if navigationAction.targetFrame == nil {
            // Handle links that open in new window/tab
            if let url = navigationAction.request.url {
                webView.load(URLRequest(url: url))
            }
        }
        decisionHandler(.allow)
    }
    
    @objc private func closePopup() {
        delegate?.offerDidClose(currentOffer.name)
        dismiss(animated: true, completion: nil)
    }
    
    @objc private func claimOffer() {
        // Action to claim the offer (open URL or perform desired action)
        if let url = URL(string: "https://apps.apple.com/redeem/?ctx=offercodes&id=\(RevenueFlo.shared.appstoreID ?? "NA")&code=\(currentOffer.customCode ?? "CODE")") {
            delegate?.offerPrimaryButtonDidClick(currentOffer.name)
            UIApplication.shared.open(url)
        }
        dismiss(animated: true, completion: nil)
    }
}

extension RevenueFloOfferVC: WKScriptMessageHandler {
  func userContentController(_ userContentController: WKUserContentController, didReceive message: WKScriptMessage) {
    if message.name == "openSafari", let urlString = message.body as? String, let url = URL(string: urlString) {
      UIApplication.shared.open(url)
    } else if message.name == "openPrimaryCTA", let urlString = message.body as? String, let url = URL(string: urlString) {
        UIApplication.shared.open(url)
        // Call delegate method for button click
        delegate?.offerPrimaryButtonDidClick(currentOffer.name)
        dismiss(animated: true, completion: nil)
    }
  }
}



extension UIColor {
    // Convenience initializer to create a UIColor from a hex string
    convenience init(hex: String) {
        var hexSanitized = hex.trimmingCharacters(in: .whitespacesAndNewlines)
        
        // Remove '#' if present
        if hexSanitized.hasPrefix("#") {
            hexSanitized.removeFirst()
        }
        
        // Default values
        var rgb: UInt64 = 0
        Scanner(string: hexSanitized).scanHexInt64(&rgb)
        
        let red = CGFloat((rgb & 0xFF0000) >> 16) / 255.0
        let green = CGFloat((rgb & 0x00FF00) >> 8) / 255.0
        let blue = CGFloat(rgb & 0x0000FF) / 255.0
        
        self.init(red: red, green: green, blue: blue, alpha: 1.0)
    }
}

public class RFPlistReader {
    
    private let plistName: String
    private let plistData: NSDictionary
    
    /// Initializes the reader with the specified plist name (without extension).
    /// - Parameter plistName: The name of the plist file (e.g., "RevenueFlo-Info").
    init(plistName: String) {
        self.plistName = plistName
        
        guard let path = Bundle.main.path(forResource: plistName, ofType: "plist"),
              let data = NSDictionary(contentsOfFile: path) else {
            assertionFailure("Error: \(plistName).plist not found or could not be loaded.")
            fatalError("Required plist \(plistName) is missing.")
        }
        
        self.plistData = data
    }
    
    /// Reads a value from the plist file.
    /// - Parameter key: The key for the value to retrieve.
    /// - Returns: The value associated with the key, or nil if not found.
    func value(forKey key: String) -> Any? {
        return plistData[key]
    }
    
    /// Reads a string value from the plist file.
    /// - Parameter key: The key for the value to retrieve.
    /// - Returns: The string value associated with the key, or nil if not found or not a string.
    func string(forKey key: String) -> String? {
        return value(forKey: key) as? String
    }
    
    /// Reads an integer value from the plist file.
    /// - Parameter key: The key for the value to retrieve.
    /// - Returns: The integer value associated with the key, or nil if not found or not an integer.
    func integer(forKey key: String) -> Int? {
        return value(forKey: key) as? Int
    }
    
    /// Reads a boolean value from the plist file.
    /// - Parameter key: The key for the value to retrieve.
    /// - Returns: The boolean value associated with the key, or nil if not found or not a boolean.
    func bool(forKey key: String) -> Bool? {
        return value(forKey: key) as? Bool
    }
}

import StoreKit

// MARK: - Customer Type Enumeration
/**
 Represents the subscription status of a user.
 */
enum CustomerType: String {
    case new = "NEW"       // User has no previous subscriptions.
    case existing = "EXISTING"  // User has an active subscription.
    case expired = "EXPIRED"   // User's subscription has expired.
    case claimed = "CLAIMED"    // User who claimed the offer already.
}

// MARK: - RFSubscriptionManager
/**
 RFSubscriptionManager is a singleton class responsible for managing and checking user subscription status.
 */
final class RFSubscriptionManager {
    
    // MARK: - Singleton Instance
    static let shared = RFSubscriptionManager()
    
    // MARK: - Initializer
    /**
     Private initializer to enforce singleton usage.
     */
    private init() {}
    
    // MARK: - Public Methods
    /**
     Checks the subscription status of the user.
     
     - Returns: A `CustomerType` value indicating the subscription status.
     */
    func checkSubscriptionStatus() async -> CustomerType {
        do {
            var hasSubscriptions = false
            let transactions = Transaction.all
            
            for await transaction in transactions {
                let payload = try transaction.payloadValue // Propagate errors
                if payload.productType == .autoRenewable {
                    hasSubscriptions = true
                }
            }
            
            if !hasSubscriptions {
                RFLogger.log("No subscriptions found.")
                return .new
            }
            
            var activeTransactions: Set<StoreKit.Transaction> = []
            for await entitlement in StoreKit.Transaction.currentEntitlements {
                let transaction = try entitlement.payloadValue // Propagate errors
                if transaction.productType == .autoRenewable {
                    activeTransactions.insert(transaction)
                }
            }
            
            if activeTransactions.isEmpty {
                RFLogger.log("No active auto-renewable transactions found.")
                return .expired
            } else {
                RFLogger.log("User has active auto-renewable transactions.")
                RevenueFlo.shared.trackOfferConversion(transaction: activeTransactions.first!)
                return .existing
            }
        } catch {
            RFLogger.log("Failed to check subscription status: \(error)", level: .error)
            return .expired
        }
    }
    
}

// MARK: - RFStorageManager
/**
 A utility class to manage secure data storage in Keychain.
 */

public class RFStorageManager {

    // MARK: - Singleton
    public static let shared = RFStorageManager()

    private init() {}

    // MARK: - Add Data to Keychain
    /// Saves a value securely in the Keychain.
    /// - Parameters:
    ///   - value: The string value to store.
    ///   - key: The key to associate with the stored value.
    /// - Returns: `true` if the operation succeeds, otherwise `false`.
    @discardableResult
    public func save(value: String, for key: String) -> Bool {
        guard let data = value.data(using: .utf8) else { return false }

        // Delete any existing item with the same key.
        delete(key: key)

        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key,
            kSecValueData as String: data
        ]

        let status = SecItemAdd(query as CFDictionary, nil)
        return status == errSecSuccess
    }

    // MARK: - Retrieve Data from Keychain
    /// Retrieves a value securely from the Keychain.
    /// - Parameter key: The key associated with the stored value.
    /// - Returns: The string value if it exists, otherwise `nil`.
    public func retrieve(for key: String) -> String? {
        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key,
            kSecReturnData as String: true,
            kSecMatchLimit as String: kSecMatchLimitOne
        ]

        var result: AnyObject?
        let status = SecItemCopyMatching(query as CFDictionary, &result)

        guard status == errSecSuccess, let data = result as? Data else {
            return nil
        }

        return String(data: data, encoding: .utf8)
    }

    // MARK: - Delete Data from Keychain
    /// Deletes a value securely from the Keychain.
    /// - Parameter key: The key associated with the value to delete.
    /// - Returns: `true` if the operation succeeds, otherwise `false`.
    @discardableResult
    public func delete(key: String) -> Bool {
        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key
        ]

        let status = SecItemDelete(query as CFDictionary)
        return status == errSecSuccess
    }

    // MARK: - Update Data in Keychain
    /// Updates an existing value securely in the Keychain.
    /// - Parameters:
    ///   - value: The new string value to store.
    ///   - key: The key associated with the value to update.
    /// - Returns: `true` if the operation succeeds, otherwise `false`.
    @discardableResult
    public func update(value: String, for key: String) -> Bool {
        guard let data = value.data(using: .utf8) else { return false }

        let query: [String: Any] = [
            kSecClass as String: kSecClassGenericPassword,
            kSecAttrAccount as String: key
        ]

        let updateAttributes: [String: Any] = [
            kSecValueData as String: data
        ]

        let status = SecItemUpdate(query as CFDictionary, updateAttributes as CFDictionary)
        return status == errSecSuccess
    }
}

func base62Encode(_ input: String) -> String {
    let base62Chars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    
    // Convert the hexadecimal input string to BInt
    guard let bIntValue = BInt(input, radix: 16) else { return "" }
    
    var value = bIntValue
    var encoded = ""
    
    // Perform Base62 encoding
    while value > 0 {
        let remainder = value % 62
        encoded.append(base62Chars[base62Chars.index(base62Chars.startIndex, offsetBy: Int(remainder))])
        value /= 62
    }
    
    // Return the reversed encoded string
    return String(encoded.reversed())
}

func base62Decode(_ input: String) -> String {
    let base62Chars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
    
    var value = BInt(0)
    
    // Decode the Base62 string to BInt
    for char in input {
        if let charIndex = base62Chars.firstIndex(of: char) {
            let digit = BInt(base62Chars.distance(from: base62Chars.startIndex, to: charIndex))
            value = value * 62 + digit
        } else {
            return ""  // Invalid character found
        }
    }
    
    // Convert the BInt back to a hexadecimal string
    return String(value, radix: 16)
}
